<?php

namespace App\Filament\Resources;

use Filament\Schemas\Schema;
use Filament\Tables\Columns\TextColumn;
use Filament\Actions\Action;
use App\Filament\Resources\SalinMaklumatTugasResource\Pages\ListSalinMaklumatTugas;
use App\Filament\Resources\SalinMaklumatTugasResource\Pages\CreateSalinMaklumatTugas;
use App\Filament\Resources\SalinMaklumatTugasResource\Pages\EditSalinMaklumatTugas;
use App\Filament\Resources\SalinMaklumatTugasResource\Pages;
use App\Filament\Resources\SalinMaklumatTugasResource\RelationManagers;
use App\Models\Sesi;
use App\Models\Pengguna;
use Filament\Forms;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\DatePicker;
use Illuminate\Database\Eloquent\Model;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Auth;

use App\Models\InstitusiJabatan;
use App\Models\Jabatan;
use App\Models\Institusi;
use App\Models\Tugasan;

class SalinMaklumatTugasResource extends Resource
{
    protected static ?string $model = Sesi::class;

    protected static string | \BackedEnum | null $navigationIcon = 'heroicon-o-rectangle-stack';

    protected static ?string $pluralModelLabel = 'SALIN MAKLUMAT TUGAS';

    protected static ?string $navigationLabel = 'SALIN MAKLUMAT TUGAS';

    protected static string | \UnitEnum | null $navigationGroup = 'UTILITI';

    protected static ?string $slug = 'salinmaklumattugas';


    public static function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Select::make('no_institusi_jabatan')
                    ->label('Institusi & Jabatan')
                    ->placeholder('Sila Pilih Maklumat')
                    ->searchable()
                    ->options(function () {
                        // Eager load related models
                        $institusiJabatans = InstitusiJabatan::with('institusi', 'jabatan')->get();

                        // Manipulate data to format as needed for select dropdown
                        return $institusiJabatans->mapWithKeys(function ($institusiJabatan) {
                            return [$institusiJabatan->no => $institusiJabatan->institusi->nama . ' - ' . $institusiJabatan->jabatan->nama];
                        });
                    }),
                Select::make('sesi_terdahulu')
                    ->label('Sesi Semasa')
                    ->placeholder('Sila Pilih Maklumat')
                    ->options(Sesi::all()->pluck('nama', 'no'))
                    ->searchable()
                    ->required(),
                Select::make('sesi_baru')
                    ->label('Sesi Baru')
                    ->placeholder('Sila Pilih Maklumat')
                    ->options(Sesi::all()->pluck('nama', 'no'))
                    ->searchable()
                    ->required(),
                DatePicker::make('tarikh_lantikan')->native(false),
            ]);

    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('index')->label('NO')->rowIndex(),
                TextColumn::make('nama')->label('SESI')->searchable(),
            ])
            ->deferLoading()
            ->filters([
                //
            ])
            ->recordActions([
              Action::make('muat_turun')
                    ->label(function ($record) {
                        $user = Auth::user();

                        $adaTugasan = Tugasan::where('no_institusi_jabatan', $user->no_institusi_jabatan)
                            ->where('sesi', $record->no)
                            ->where('aktif', 1)
                            ->exists();

                        return $adaTugasan ? 'SALIN' : '⚠️ TIADA DATA TUGASAN. SILA SALIN DARI SESI TERDAHULU.';
                    })
                    ->icon(function ($record) {
                        $user = Auth::user();

                        $adaTugasan = Tugasan::where('no_institusi_jabatan', $user->no_institusi_jabatan)
                            ->where('sesi', $record->no)
                            ->where('aktif', 1)
                            ->exists();

                        return $adaTugasan ? 'heroicon-s-document-minus' : null;
                    })
                    ->disabled(function ($record) {
                        $user = Auth::user();

                        // Disable button kalau tiada data
                        return ! Tugasan::where('no_institusi_jabatan', $user->no_institusi_jabatan)
                            ->where('sesi', $record->no)
                            ->where('aktif', 1)
                            ->exists();
                    })
                    ->schema(function ($record) {
                        $formFields = [];

                        // Assuming $record->no is used to determine the form fields dynamically
                        switch ($record->no) {
                            case 1:
                            //
                            // break;
                            case 2:
                            //
                            // break;
                            case 3:
                            //
                            // break;
                            case 4:
                            //
                            // break;
                            case 5:
                            //
                            // break;
                            // Add more cases as needed

                            default:
                                // Default behavior if $record->no doesn't match any case

                                $user = Auth::user();
                                $adaTugasan = Tugasan::where('no_institusi_jabatan', $user->no_institusi_jabatan)
                                    ->where('sesi', $record->no)
                                    ->where('aktif', 1)
                                    ->exists();

                                $disabled = ! $adaTugasan;

                                $formFields[] = Select::make('sesi_semasa')
                                    ->label('Sesi Semasa')
                                    ->options(Sesi::where('no', $record->no)->pluck('nama', 'no'))
                                    ->default($record->no)
                                    ->required()
                                    ->dehydrated(true)
                                    ->disabled($disabled)
                                    ->helperText(function () use ($record) {

                                        $user = Auth::user();

                                        $adaTugasan = Tugasan::where('no_institusi_jabatan', $user->no_institusi_jabatan)
                                            ->where('sesi', $record->no)
                                            ->where('aktif', 1)
                                            ->exists();

                                        return $adaTugasan
                                            ? ''
                                            : '⚠️ Tiada data tugasan untuk sesi ini. Sila pilih sesi yang mempunyai data tugasan.';
                                    });

                                $formFields[] = Select::make('sesi_baru')
                                    ->label('Sesi Baru')
                                    ->options(Sesi::whereNot('no', $record->no)->pluck('nama', 'no')) // Assuming Sesi is a model
                                    ->searchable()
                                    ->disabled($disabled)
                                    ->required();
                                $formFields[] = Select::make('no_institusi_jabatan')
                                    ->label('Institusi & Jabatan')
                                    ->placeholder('Sila Pilih Maklumat')
                                    ->disabled($disabled)
                                    ->options(function () {
                                        // Eager load related models

                                        $jenis_pengguna = Auth::user()->jenis_pengguna;
                                        $email = Auth::user()->email;
                                        $no_institusi_jabatan = Auth::user()->no_institusi_jabatan;

                                        $institusiJabatans = InstitusiJabatan::with('institusi', 'jabatan');

                                        if ($jenis_pengguna == 2) {
                                            $userId = Auth::id();
                                            // Modify the query to include a where clause for jenis_pengguna = 2
                                            $institusiJabatans->where('no', $no_institusi_jabatan);
                                        }

                                        $institusiJabatans = $institusiJabatans->get();

                                        // Manipulate data to format as needed for select dropdown
                                        return $institusiJabatans->mapWithKeys(function ($institusiJabatan) {
                                            return [$institusiJabatan->no => $institusiJabatan->institusi->nama . ' - ' . $institusiJabatan->jabatan->nama];
                                        });
                                    })
                                    ->searchable()
                                    ->required(); // Assuming Sesi is a model
                                $formFields[] = DatePicker::make('tarikh_lantikan')->label('Tarikh Lantikan')->disabled($disabled)->required()->native(false);

                                break;
                        }

                        return $formFields;
                    })
                    ->action(function (array $data, Sesi $record) {
                        $senarai_tugas_sampingan = Tugasan::orderBy('no', 'asc')->where('no_institusi_jabatan', $data['no_institusi_jabatan'])->where('sesi', $data['sesi_semasa'])->where('aktif', 1)->get();

                        foreach ($senarai_tugas_sampingan as $tugas_sampingan) {

                            $semakPenduaTugas = Tugasan::where('email', $tugas_sampingan->email)->where('no_tugas_sampingan', $tugas_sampingan->no_tugas_sampingan)->where('no_unit_tugas_sampingan', $tugas_sampingan->no_unit_tugas_sampingan)->where('peringkat', $tugas_sampingan->peringkat)->where('no_institusi_jabatan', $tugas_sampingan->no_institusi_jabatan)->where('tarikh_lantikan', date("Y-m-d", strtotime($tugas_sampingan->tarikh_lantikan)))->where('sesi', $data['sesi_baru'])->where('aktif', 1)->count();

                            if ($semakPenduaTugas > 0) {

                            } else {

                                $emailTerkini = Pengguna::where('email', $tugas_sampingan->email)->first();

                                if(is_null($emailTerkini)) {

                                } else {

                                     Tugasan::create([
                                    'email' => $tugas_sampingan->email,
                                    'gred' => $emailTerkini->gred,
                                    'no_tugas_sampingan' => $tugas_sampingan->no_tugas_sampingan,
                                    'no_unit_tugas_sampingan' => $tugas_sampingan->no_unit_tugas_sampingan,
                                    'peringkat' => $tugas_sampingan->peringkat,
                                    'no_institusi_jabatan' => $tugas_sampingan->no_institusi_jabatan,
                                    'sesi' => $data['sesi_baru'],
                                    'tarikh_lantikan' => date("Y-m-d", strtotime($data['tarikh_lantikan'])),
                                    'no_status' => $tugas_sampingan->no_status,
                                    'catatan' => strtoupper($tugas_sampingan->catatan),
                                    'aktif' => 1,
                                ]);

                                }

                            }


                        }
                        // Send success notification
                        Notification::make()
                            ->title('Maklumat Tugas berjaya disalin.')
                            ->success()
                            ->send();
                    }),
            ])
            ->toolbarActions([
                //
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListSalinMaklumatTugas::route('/'),
            'create' => CreateSalinMaklumatTugas::route('/create'),
            'edit' => EditSalinMaklumatTugas::route('/{record:slug}/edit'),
        ];
    }

    // Override the save method to customize how data is saved
    public function save(Schema $schema, $record)
    {
        // Perform any additional operations before saving if needed




        // Save the record using Eloquent ORM
        $record->save();

        // Perform any additional operations after saving if needed

        // Return the saved record
        return $record;
    }

    protected function handleRecordCreation(array $data): Model
    {
        return static::getModel()::create($data);
    }

    public static function canCreate(): bool
    {
        return false;
    }

    public static function canViewAny(): bool
    {
        // dd(auth()->user()->roles());
        return auth()
            ->user()
            ->hasAnyRole(['Super Admin', 'Penyelaras']);
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::count();
    }

    public static function canEdit(Model $record): bool
    {
        return false;
    }


}
