<?php

namespace App\Filament\Resources;

use App\Filament\Resources\TugasanResource\Pages;
use App\Filament\Resources\TugasanResource\RelationManagers;
use App\Models\Tugasan;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Columns\TextColumn;
use Illuminate\Support\Facades\Hash;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Tables\Columns\IconColumn;
use Filament\Forms\Components\FileUpload;
use Filament\Tables\Columns\ImageColumn;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Textarea;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Support\Facades\Auth;
use Illuminate\Contracts\Pagination\Paginator;

use App\Models\User;
use App\Models\Gred;
use App\Models\JenisPengguna;
use App\Models\InstitusiJabatan;
use App\Models\Jabatan;
use App\Models\Institusi;
use App\Models\Peringkat;
use App\Models\Sesi;
use App\Models\Status;
use App\Models\TugasSampingan;
use App\Models\UnitTugasSampingan;

class TugasanResource extends Resource
{
    protected static ?string $model = Tugasan::class;

    protected static ?string $navigationIcon = 'heroicon-s-document-text';

    protected static ?string $navigationLabel = 'TUGASAN';

    protected static ?string $pluralModelLabel = 'TUGASAN';

    protected static ?string $navigationGroup = 'TUGASAN';

    protected static ?string $slug = 'tugasan';

    public static function form(Form $form): Form
    {
        $jenis_pengguna = auth()->user()->roles()->value('id');
        $email = Auth::user()->email;
        $no_institusi_jabatan = Auth::user()->no_institusi_jabatan;

        return $form->schema([
            Select::make('email')
                ->label('Pengguna')
                ->placeholder('Sila Pilih Maklumat')
                ->options(function () use ($jenis_pengguna, $no_institusi_jabatan) {
                    // Eager load related models

                    $pengguna = User::with('institusiJabatan')->where('aktif', 1);

                    if ($jenis_pengguna == 2) {
                        $userId = Auth::id();
                        // Modify the query to include a where clause for jenis_pengguna = 2
                        $pengguna->where('no_institusi_jabatan', $no_institusi_jabatan);
                    }

                    $pengguna = $pengguna->orderBy('nama')->get();

                    // Manipulate data to format as needed for select dropdown
                    return $pengguna->mapWithKeys(function ($pengguna) {
                        return [$pengguna->email => $pengguna->nama];
                    });
                })
                ->reactive()
                ->afterStateUpdated(function (callable $set, $state) {
                    $pengguna = User::where('email', $state)->first();
                    if ($pengguna) {
                        $set('gred', $pengguna->gred); // Auto-set gred (assuming pengguna->gred holds the ID)
                    }
                })
                ->searchable()
                ->rules('required'),
            Select::make('gred')
                ->label('Gred')
                ->placeholder('Sila Pilih Maklumat')
                ->options(Gred::where('aktif', 1)->orderBy('nama')->pluck('nama', 'no'))
                ->searchable()
                ->rules('required'),
            Select::make('no_tugas_sampingan')
                ->label('Tugas Sampingan')
                ->placeholder('Sila Pilih Maklumat')
                ->options(TugasSampingan::where('aktif', 1)->orderBy('nama')->pluck('nama', 'no'))
                ->searchable()
                ->rules('required'),
            Select::make('no_unit_tugas_sampingan')
                ->label('Unit Tugas Sampingan')
                ->placeholder('Sila Pilih Maklumat')
                ->options(UnitTugasSampingan::where('aktif', 1)->pluck('nama', 'no'))
                ->default(1)
                ->searchable()
                ->rules('required'),
            Select::make('peringkat')
                ->label('Peringkat')
                ->placeholder('Sila Pilih Maklumat')
                ->options(Peringkat::all()->pluck('nama', 'no'))
                ->searchable()
                ->rules('required'),
            Select::make('no_institusi_jabatan')
                ->label('Institusi & Jabatan')
                ->placeholder('Sila Pilih Maklumat')
                ->searchable()
                ->default(function () use ($jenis_pengguna) {
                    if ($jenis_pengguna == 2) {
                        return Auth::user()->no_institusi_jabatan;
                    }
                })
                ->options(function () use ($jenis_pengguna, $no_institusi_jabatan) {
                    // Eager load related models

                    $institusiJabatans = InstitusiJabatan::with('institusi', 'jabatan');

                    if ($jenis_pengguna == 2) {
                        $userId = Auth::id();
                        // Modify the query to include a where clause for jenis_pengguna = 2
                        $institusiJabatans->where('no', $no_institusi_jabatan);
                    }

                    $institusiJabatans = $institusiJabatans->get();

                    // Manipulate data to format as needed for select dropdown
                    return $institusiJabatans->mapWithKeys(function ($institusiJabatan) {
                        return [$institusiJabatan->no => $institusiJabatan->institusi->nama . ' - ' . $institusiJabatan->jabatan->nama];
                    });
                })
                ->rules('required'),
            DatePicker::make('tarikh_lantikan')->native(false)->rules('required'),
            Select::make('sesi')
                ->label('Sesi')
                ->placeholder('Sila Pilih Maklumat')
                ->options(Sesi::all()->pluck('nama', 'no'))
                ->searchable()
                ->rules('required'),
            Select::make('no_status')
                ->label('Status')
                ->placeholder('Sila Pilih Maklumat')
                ->options(Status::all()->pluck('nama', 'no'))
                ->searchable()
                ->default(3)
                ->rules('required'),
            Select::make('aktif')
                ->label('Status')
                ->placeholder('Sila Pilih Maklumat')
                ->options([
                    '1' => 'AKTIF',
                    '0' => 'TIDAK AKTIF',
                ])
                ->rules('required'),
            Textarea::make('catatan')
                ->autosize()
                ->rules(['max:500']),
        ]);
    }

    public static function table(Table $table): Table
    {
        $jenis_pengguna = Auth::user()->jenis_pengguna;
        $email = Auth::user()->email;
        $no_institusi_jabatan = Auth::user()->no_institusi_jabatan;

        // Adjust the query based on the user's role
        $tugasQuery = Tugasan::query()
            ->where('aktif', 1)
            ->with(['pengguna', 'tugasSampingan', 'unitTugasSampingan', 'senaraiPeringkat', 'senaraisesi', 'status']);
        if ($jenis_pengguna == 2 || $jenis_pengguna == 3 || $jenis_pengguna == 4) {
            // Otherwise, filter jenis laporan based on user's role
            $tugasQuery->whereHas('pengguna', function ($query) use ($no_institusi_jabatan) {
                $query->where('no_institusi_jabatan', $no_institusi_jabatan);
            });
        }

        return $table
            ->query($tugasQuery)
            ->columns([
                Tables\Columns\TextColumn::make('index')->label('NO')->rowIndex(),
                Tables\Columns\TextColumn::make('pengguna.nama')->label('Nama'),
                Tables\Columns\TextColumn::make('tugasSampingan.nama')->label('Tugas Sampingan'),
                Tables\Columns\TextColumn::make('unitTugasSampingan.nama')->label('Unit Tugas Sampingan'),
                Tables\Columns\TextColumn::make('senaraiPeringkat.nama')->label('Peringkat'),
                Tables\Columns\TextColumn::make('senaraisesi.nama')->label('Sesi'),
                Tables\Columns\TextColumn::make('status.nama')->label('STATUS')->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('pengguna.senaraigred.nama')->label('Gred'),
                Tables\Columns\TextColumn::make('pengguna.institusiJabatan.institusi.nama')->label('Institusi'),
                Tables\Columns\TextColumn::make('pengguna.institusiJabatan.jabatan.nama')->label('Jabatan'),
                Tables\Columns\TextColumn::make('tarikh_lantikan')->label('Tarikh Lantikan'),
                Tables\Columns\TextColumn::make('catatan')->label('Catatan'),
                IconColumn::make('aktif')->label('PENGAKTIFAN')->boolean()->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('pengguna.nama')->label('NAMA')->searchable(),
                TextColumn::make('tugasSampingan.nama')->label('TUGAS SAMPINGAN')->searchable(),
                TextColumn::make('unitTugasSampingan.nama')->label('UNIT TUGAS SAMPINGAN')->searchable()->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('senaraiPeringkat.nama')->label('PERINGKAT')->searchable(),
                TextColumn::make('senaraisesi.nama')->label('SESI')->searchable(),
                TextColumn::make('pengguna.senaraigred.nama')->label('GRED')->searchable()->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('pengguna.institusiJabatan.institusi.nama')->label('INSTITUSI')->searchable()->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('pengguna.institusiJabatan.jabatan.nama')->label('JABATAN')->searchable()->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('tarikh_lantikan')->label('TARIKH LANTIKAN')->searchable()->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('catatan')->label('CATATAN')->searchable()->toggleable(isToggledHiddenByDefault: true),
            ])
            ->defaultSort(fn($query) => $query->orderBy('sesi', 'desc')->orderBy('email', 'asc')->orderBy('peringkat', 'desc'))
            ->deferLoading()
            ->striped()
            ->filters([
                SelectFilter::make('sesi')->relationship('senaraiSesi', 'nama')->searchable()->preload(),
                SelectFilter::make('no_institusi_jabatan')
                    ->options(function () use ($jenis_pengguna, $no_institusi_jabatan) {
                        // Eager load relationships and filter data
                        $query = User::with('institusiJabatan.jabatan');

                        // Apply additional condition based on jenis_pengguna
                        if ($jenis_pengguna == 1) {
                            // You can add any specific condition for jenis_pengguna == 1 here if needed
                        } elseif ($jenis_pengguna == 4) {
                            $institusi_jabatan = User::with('institusiJabatan')->where('no_institusi_jabatan', $no_institusi_jabatan)->first();
                            if ($institusi_jabatan && $institusi_jabatan->institusiJabatan) {
                                $no_institusi = $institusi_jabatan->institusiJabatan->no_institusi;
                                $query->whereHas('institusiJabatan', function ($q) use ($no_institusi) {
                                    $q->where('no_institusi', $no_institusi);
                                });
                            }
                        }

                        return $query
                            ->get()
                            ->filter(function ($pengguna) {
                                // Ensure both institusiJabatan and jabatan are not null
                                return $pengguna->institusiJabatan && $pengguna->institusiJabatan->jabatan;
                            })
                            ->mapWithKeys(function ($pengguna) {
                                // Map correct fields
                                return [
                                    $pengguna->institusiJabatan->jabatan->no => $pengguna->institusiJabatan->jabatan->nama,
                                ];
                            })
                            ->all(); // Convert collection to array
                    })
                    ->label('Jabatan')
                    ->multiple()
                    ->visible(fn() => auth()->user()->jenis_pengguna == 1 || auth()->user()->jenis_pengguna == 4),
                SelectFilter::make('email')
                    ->label('Pengguna')
                    ->relationship('pengguna', 'nama', function (Builder $query) use ($no_institusi_jabatan) {
                        // Filter the query to include only users belonging to the same institusi jabatan
                        $query->whereHas('institusiJabatan', function (Builder $subQuery) use ($no_institusi_jabatan) {
                            $subQuery->where('no', $no_institusi_jabatan);
                        });
                    })
                    ->multiple()
                    ->searchable()
                    ->preload(),
                SelectFilter::make('peringkat')->relationship('senaraiPeringkat', 'nama')->searchable()->preload(),
            ])
            ->actions([Tables\Actions\EditAction::make()->label(''), Tables\Actions\DeleteAction::make()->label('')])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    //Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
                //
            ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTugasans::route('/'),
            'create' => Pages\CreateTugasan::route('/create'),
            'edit' => Pages\EditTugasan::route('/{record:slug}/edit'),
        ];
    }

    public static function canViewAny(): bool
    {
        // dd(auth()->user()->roles());
        return auth()
            ->user()
            ->hasAnyRole(['Super Admin', 'Penyelaras']);
    }

    protected function paginateTableQuery(Builder $query): Paginator
    {
        return $query->fastPaginate($this->getTableRecordsPerPage());
    }

    public static function getNavigationBadge(): ?string
    {
        $user = Auth::user(); // Get the authenticated user

        if (!$user) {
            return null; // Return null if no user is logged in
        }

        if ($user->jenis_pengguna == 1) {
            // Role 1: Return the total count of all records
            return static::getModel()::count();
        } elseif ($user->jenis_pengguna == 2) {
            // Role 2: Count records filtered by the user's institusi jabatan
            return static::getModel()
                ::whereHas('pengguna', function ($query) use ($user) {
                    $query->where('no_institusi_jabatan', $user->no_institusi_jabatan)->where('tugas.aktif',1);
                })
                ->count();
        }

        return null; // Default for other roles
    }
}
