<?php

namespace App\Filament\Resources;

use App\Filament\Resources\SemakanPemberatResource\Pages;
use App\Filament\Resources\SemakanPemberatResource\RelationManagers;
use App\Models\Tugasan;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\Summarizers\Sum;
use Filament\Tables\Columns\Summarizers\Count;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Support\Facades\Auth;
use Illuminate\Contracts\Pagination\Paginator;

use App\Models\Pengguna;
use App\Models\InstitusiJabatan;

class SemakanPemberatResource extends Resource
{
    protected static ?string $model = Tugasan::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    protected static ?string $pluralModelLabel = 'SEMAKAN PEMBERAT';

    protected static ?string $navigationLabel = 'SEMAKAN PEMBERAT';

    protected static ?string $navigationGroup = 'LAPORAN';

    protected static ?string $slug = 'semakanpemberat';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function table(Table $table): Table
    {

        $jenis_pengguna = auth()->user()->roles()->value('id');
        $email = Auth::user()->email;
        $no_institusi_jabatan = Auth::user()->no_institusi_jabatan;

        // Adjust the query based on the user's role
        $tugasQuery = Tugasan::query()
            //->whereHas('pengguna')
            ->with(['pengguna', 'senaraiSesi', 'senaraiPeringkat'])->where('aktif',1);
        if ($jenis_pengguna == 1) {
            // If user is not admin, filter jenis laporan based on user's role
            // You can add your filtering logic here
        } elseif ($jenis_pengguna == 2) {
            // If user is of type 2, filter based on no_institusi_jabatan
            $tugasQuery->whereHas('pengguna', function ($query) use ($no_institusi_jabatan) {
                $query->where('no_institusi_jabatan', $no_institusi_jabatan);
            });
        } elseif ($jenis_pengguna == 4) {
            // If user is of type 4, exclude specific jenis laporan
            $institusi_jabatan = InstitusiJabatan::where('no', $no_institusi_jabatan)->first();
            $institusi = $institusi_jabatan->no_institusi;

            $tugasQuery->whereHas('pengguna.institusiJabatan', function ($query) use ($institusi) {
                $query->where('no_institusi', $institusi);
            });
        }

        return $table
            ->query($tugasQuery)
            ->columns([
                Tables\Columns\TextColumn::make('pengguna.nama'),
                Tables\Columns\TextColumn::make('pemberat')
                    ->label('PEMBERAT')
                    ->summarize(Sum::make()),
                TextColumn::make('pengguna.nama')->label('NAMA')->searchable(isIndividual: true, isGlobal: false),
            ])
            ->defaultGroup('pengguna.nama')
            ->groupsOnly()
            ->deferLoading()
            ->striped()
            ->filters([
                SelectFilter::make('sesi')
                    ->relationship('senaraiSesi', 'nama')
                    ->searchable()
                    ->preload(),
                SelectFilter::make('no_institusi_jabatan')
                    ->options(function () use ($jenis_pengguna, $no_institusi_jabatan) {
                        // Eager load relationships and filter data
                        $query = Pengguna::with('institusiJabatan.jabatan');

                        // Apply additional condition based on jenis_pengguna
                        if ($jenis_pengguna == 1) {
                            // You can add any specific condition for jenis_pengguna == 1 here if needed
                        } elseif ($jenis_pengguna == 4) {
                            $institusi_jabatan = Pengguna::with('institusiJabatan')->where('no_institusi_jabatan', $no_institusi_jabatan)->first();
                            if ($institusi_jabatan && $institusi_jabatan->institusiJabatan) {
                                $no_institusi = $institusi_jabatan->institusiJabatan->no_institusi;
                                $query->whereHas('institusiJabatan', function ($q) use ($no_institusi) {
                                    $q->where('no_institusi', $no_institusi);
                                });
                            }
                        }

                        return $query->get()
                            ->filter(function ($pengguna) {
                                // Ensure both institusiJabatan and jabatan are not null
                                return $pengguna->institusiJabatan && $pengguna->institusiJabatan->jabatan;
                            })
                            ->mapWithKeys(function ($pengguna) {
                                // Map correct fields
                                return [
                                    $pengguna->institusiJabatan->jabatan->no => $pengguna->institusiJabatan->jabatan->nama
                                ];
                            })
                            ->all(); // Convert collection to array
                    })
                    ->label('Jabatan')
                    ->multiple(),
                SelectFilter::make('peringkat')
                    ->relationship('senaraiPeringkat', 'nama')
                    ->searchable()
                    ->preload(),
            ])
            ->actions([
                //
            ])
            ->bulkActions([
                //
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListSemakanPemberats::route('/'),
            // 'create' => Pages\CreateSemakanPemberat::route('/create'),
            // 'edit' => Pages\EditSemakanPemberat::route('/{record}/edit'),
        ];
    }

    public static function canCreate(): bool
    {
        return false;
    }

    public static function canViewAny(): bool
    {
        // dd(auth()->user()->roles());
        return auth()
            ->user()
            ->hasAnyRole(['Super Admin', 'Penyelaras','Pengurusan']);
    }

    protected function paginateTableQuery(Builder $query): Paginator
    {
        return $query->fastPaginate($this->getTableRecordsPerPage());
    }
}
