<?php

namespace App\Filament\Resources;

use App\Filament\Resources\DokumenRujukanResource\Pages;
use App\Filament\Resources\DokumenRujukanResource\RelationManagers;
use App\Models\DokumenRujukan;
use App\Models\JenisDokumen;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\Select;
use Illuminate\Support\Facades\Auth;

use App\Models\User;
use App\Models\Sesi;
use App\Models\JenisLaporan;
use App\Models\InstitusiJabatan;


class DokumenRujukanResource extends Resource
{
    protected static ?string $model = DokumenRujukan::class;

    protected static ?string $navigationIcon = 'heroicon-s-document-text';

    protected static ?string $navigationLabel = 'DOKUMEN RUJUKAN';

    protected static ?string $pluralModelLabel = 'DOKUMEN RUJUKAN';

    protected static ?string $navigationGroup = 'TUGASAN';

    protected static ?string $slug = 'dokumenrujukan';

    public static function form(Form $form): Form
    {
        $jenis_pengguna = auth()->user()->roles()->value('id');;
        $email = Auth::user()->email;
        $no_institusi_jabatan = Auth::user()->no_institusi_jabatan;

        return $form
            ->schema([
                Select::make('jenis_dokumen_rujukan')
                    ->label('Jenis Dokumen')
                    ->placeholder('Sila Pilih Maklumat')
                    ->options(function () use ($jenis_pengguna, $no_institusi_jabatan) {
                        // Eager load related models

                        $jenislaporan = JenisLaporan::query()->where('aktif', 1);

                        if ($jenis_pengguna == 2) {
                            // Modify the query to include a where clause for jenis_pengguna = 2
                            $jenislaporan->where('no', 4);
                        } elseif ($jenis_pengguna == 4) {
                            $jenislaporan->whereIn('no', [4, 5]);
                        }

                        $jenislaporan = $jenislaporan->orderBy('nama')->get();

                        // Manipulate data to format as needed for select dropdown
                        return $jenislaporan->mapWithKeys(function ($jenislaporan) {
                            return [$jenislaporan->no => $jenislaporan->nama];
                        });
                    })
                    ->searchable()
                    ->rules('required')
                    ->live(),
                Forms\Components\TextInput::make('nama')->label('Nama')->rules(['required', 'max:200']),
                Forms\Components\TextInput::make('no_dokumen')->label('No. Dokumen')->integer()->rules('required')->visible(
                    fn($record, $get) => $get('jenis_dokumen_rujukan') === '4'
                ),
                Select::make('no_sesi')
                    ->label('Sesi')
                    ->placeholder('Sila Pilih Maklumat')
                    ->options(Sesi::all()->where('aktif', 1)->pluck('nama', 'no'))
                    ->searchable()
                    ->rules('required'),
                Select::make('email_ketua_jabatan')
                    ->label('Pengarah/Ketua Jabatan')
                    ->placeholder('Sila Pilih Maklumat')
                    ->options(function () use ($jenis_pengguna, $no_institusi_jabatan) {
                        // Eager load related models

                        $pengguna = User::with('institusiJabatan')->where('aktif', 1);
                        $institusi_jabatan = InstitusiJabatan::where('no', $no_institusi_jabatan)->first();
                        $institusi = $institusi_jabatan->no_institusi;

                        if ($jenis_pengguna == 2) {
                            // Modify the query to include a where clause for jenis_pengguna = 2
                            $pengguna->where('no_institusi_jabatan', $no_institusi_jabatan);
                        } else if ($jenis_pengguna == 4) {
                            $pengguna->whereHas('institusiJabatan', function ($query) use ($institusi) {
                                $query->where('no_institusi', $institusi);
                            });
                        }

                        $pengguna = $pengguna->orderBy('nama')->get();

                        // Manipulate data to format as needed for select dropdown
                        return $pengguna->mapWithKeys(function ($pengguna) {
                            return [$pengguna->email => $pengguna->nama];
                        });
                    })
                    ->searchable()
                    ->rules('required'),
                Select::make('aktif')
                    ->placeholder('Sila Pilih Maklumat')
                    ->options([
                        '1' => 'AKTIF',
                        '0' => 'TIDAK AKTIF',
                    ])
                    ->rules('required'),
            ]);
    }

    public static function table(Table $table): Table
    {
        $jenis_pengguna = Auth::user()->jenis_pengguna;
        $email = Auth::user()->email;
        $no_institusi_jabatan = Auth::user()->no_institusi_jabatan;

        // Adjust the query based on the user's role
        $dokumenRujukanQuery = DokumenRujukan::query()
            ->with(['pengguna', 'jenislaporan', 'sesi']);

        if ($jenis_pengguna == 1) {
            // If user is admin, show all
        } elseif ($jenis_pengguna == 2) {
            // Otherwise, filter jenis laporan based on user's role
            $dokumenRujukanQuery->whereHas('pengguna', function ($query) use ($no_institusi_jabatan) {
                $query->where('no_institusi_jabatan', $no_institusi_jabatan);
            });
        } else {
            $institusi_jabatan_individu = InstitusiJabatan::where('no', $no_institusi_jabatan)->first();
            $institusi = $institusi_jabatan_individu->no_institusi;

            $dokumenRujukanQuery->whereHas('pengguna.institusiJabatan', function ($query) use ($institusi) {
                $query->where('no_institusi', $institusi)->where('jenis_dokumen_rujukan', 5);
            });
        }

        return $table
            ->query($dokumenRujukanQuery)
            ->columns([
                Tables\Columns\TextColumn::make('index')->label('No')->rowIndex(),
                Tables\Columns\TextColumn::make('nama'),
                Tables\Columns\TextColumn::make('sesi.nama'),
                Tables\Columns\TextColumn::make('jenislaporan.nama'),
                Tables\Columns\TextColumn::make('no_dokumen'),
                Tables\Columns\TextColumn::make('pengguna.nama'),
                IconColumn::make('aktif')
                    ->label('STATUS')
                    ->boolean(),
                TextColumn::make('nama')->label('NAMA DOKUMEN')->searchable(),
                TextColumn::make('sesi.nama')->label('SESI')->searchable(),
                TextColumn::make('jenislaporan.nama')->label('JENIS DOKUMEN')->searchable(),
                TextColumn::make('no_dokumen')->label('NO. DOKUMEN')->searchable(),
                TextColumn::make('pengguna.nama')->label('PENGARAH/KETUA JABATAN')->searchable(),
            ])
            ->deferLoading()
            ->striped()
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make()->label(''),
                Tables\Actions\DeleteAction::make()->label(''),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    //Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListDokumenRujukans::route('/'),
            'create' => Pages\CreateDokumenRujukan::route('/create'),
            'edit' => Pages\EditDokumenRujukan::route('/{record:slug}/edit'),
        ];
    }
    public static function canViewAny(): bool
    {
        $userId = auth()->user()->jenis_pengguna;
        return in_array($userId, [1, 2, 4]);
    }

    public static function getNavigationBadge(): ?string
    {
        $user = Auth::user(); // Get the authenticated user

        if (!$user) {
            return null; // Return null if no user is logged in
        }

        if ($user->jenis_pengguna == 1) {
            // Role 1: Return the total count of all records
            return static::getModel()::count();
        } elseif ($user->jenis_pengguna == 2) {
            // Role 2: Count records filtered by the user's institusi jabatan
            return static::getModel()::whereHas('pengguna', function ($query) use ($user) {
                $query->where('no_institusi_jabatan', $user->no_institusi_jabatan);
            })->count();
        }

        return null; // Default for other roles
    }

}
