<?php

namespace App\Models;

use Illuminate\Support\Facades\Storage;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;
use Spatie\Permission\Traits\HasRoles;
use Althinect\FilamentSpatieRolesPermissions\Concerns\HasSuperAdmin;

class Pengguna extends Model implements Auditable
{
    use HasFactory;
    use AuditableTrait;
    use HasRoles;
    use HasSuperAdmin;

    protected $table = 'users';

    protected $fillable = ['email', 'password', 'nama', 'gambar', 'gred', 'no_institusi_jabatan', 'aktif', 'jenis_pengguna'];

    protected static function boot()
    {
        parent::boot();

        // Define an event listener for the creating event
        static::creating(function ($pengguna) {
            // Modify the 'nama' attribute to uppercase before insertion
            $pengguna->nama = strtoupper($pengguna->nama);
        });

        static::updating(function ($pengguna) {
            // Retrieve the original model instance before update

            $pengguna->nama = strtoupper($pengguna->nama);
        });
    }

    public function institusiJabatan()
    {
        return $this->belongsTo(InstitusiJabatan::class, 'no_institusi_jabatan', 'no');
    }

    public function jenisPengguna()
    {
        return $this->belongsTo(JenisPengguna::class, 'jenis_pengguna', 'no');
    }

    public function dokumenRujukan()
    {
        return $this->hasMany(DokumenRujukan::class, 'email_ketua_jabatan', 'email');
    }

    public function senaraiGred()
    {
        return $this->belongsTo(Gred::class, 'gred', 'no');
    }

    public function tugasan()
    {
        return $this->hasMany(Tugasan::class, 'email', 'email');
    }

    public static function senaraiPengguna($institusi, $jabatan)
    {
        $senarai_pengguna = Pengguna::where('users.aktif', 1)->leftJoin('institusi_jabatan', 'institusi_jabatan.no', '=', 'users.no_institusi_jabatan')->leftJoin('jabatan', 'jabatan.no', '=', 'institusi_jabatan.no_jabatan')->leftJoin('gred', 'gred.no', '=', 'users.gred')->leftJoin('institusi', 'institusi.no', '=', 'institusi_jabatan.no_institusi')->leftJoin('jenis_pengguna', 'jenis_pengguna.no', '=', 'users.jenis_pengguna')->select('users.email as email', 'users.nama as nama', 'users.gred as gred', 'institusi_jabatan.no_jabatan as jabatan', 'users.aktif as aktif', 'users.jenis_pengguna as jenis_pengguna', 'institusi_jabatan.unit_tugas_sampingan as unit_tugas_sampingan', 'institusi_jabatan.no_institusi as no_institusi', 'gred.nama as nama_gred', 'institusi.nama as nama_institusi', 'jenis_pengguna.nama as nama_jenis_pengguna', 'users.no_institusi_jabatan as no_institusi_jabatan', 'jabatan.nama as nama_jabatan')->where('institusi.no', $institusi);

        if (!empty($jabatan)) {
            $senarai_pengguna->where('jabatan.no', $jabatan);
        }

        $senarai_pengguna = $senarai_pengguna->orderBy('users.nama')->get();

        return $senarai_pengguna;
    }

    public function getRouteKeyName(): string
    {
        return 'slug'; // Laravel will now use slug for route model binding
    }

}
