<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Schema;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class InstallerController extends Controller
{
    public function install()
    {
        if (File::exists(storage_path('installed.lock'))) {
            return response()->json([
                'status' => 'error',
                'message' => 'Application already installed.'
            ]);
        }

        try {
            // --------------------------
            // 1️⃣ CREATE TABLES (Schema changes outside transaction)
            // --------------------------
            if (!Schema::hasTable('users')) {
                Schema::create('users', function (Blueprint $table) {
                    $table->id();
                    $table->string('name');
                    $table->string('email')->unique();
                    $table->string('password');
                    $table->tinyInteger('jenis_pengguna')->default(3);
                    $table->timestamps();
                });
            }

            if (!Schema::hasTable('roles')) {
                Schema::create('roles', function (Blueprint $table) {
                    $table->id();
                    $table->string('name');
                    $table->string('guard_name')->default('web');
                    $table->timestamps();
                });
            }

            if (!Schema::hasTable('permissions')) {
                Schema::create('permissions', function (Blueprint $table) {
                    $table->id();
                    $table->string('name');
                    $table->string('guard_name')->default('web');
                    $table->timestamps();
                });
            }

            if (!Schema::hasTable('role_has_permissions')) {
                Schema::create('role_has_permissions', function (Blueprint $table) {
                    $table->unsignedBigInteger('permission_id');
                    $table->unsignedBigInteger('role_id');
                    $table->primary(['permission_id', 'role_id']);
                });
            }

            if (!Schema::hasTable('model_has_roles')) {
                Schema::create('model_has_roles', function (Blueprint $table) {
                    $table->unsignedBigInteger('role_id');
                    $table->string('model_type');
                    $table->unsignedBigInteger('model_id');
                    $table->index(['model_id', 'model_type']);
                });
            }

            // --------------------------
            // 2️⃣ ADD/UPDATE SLUGS (also outside transaction)
            // --------------------------
            $tables = [
                'dokumen_rujukan', 'gred', 'institusi', 'institusi_jabatan', 'jabatan', 'jawatan',
                'jenis_dokumen', 'jenis_dokumen_rujukan', 'jenis_laporan', 'jenis_pengguna',
                'menu', 'menu_sub_menu', 'pautan_menu_sub_menu', 'peringkat', 'senarai_jpka',
                'sesi', 'status', 'sub_menu', 'tugas', 'tugas_sampingan', 'unit_tugas_sampingan', 'users'
            ];

            foreach ($tables as $table) {
                if (!Schema::hasColumn($table, 'slug')) {
                    Schema::table($table, function (Blueprint $t) {
                        $t->string('slug', 150)->nullable();
                    });
                } else {
                    Schema::table($table, function (Blueprint $t) {
                        $t->string('slug', 150)->nullable()->change();
                    });
                }

                $rows = DB::table($table)->get();
                foreach ($rows as $row) {
                    $slug = substr(str_replace('-', '', Str::uuid()) . Str::upper(Str::random(68)), 0, 150);
                    if (isset($row->id)) {
                        DB::table($table)->where('id', $row->id)->update(['slug' => $slug]);
                    } elseif (isset($row->no)) {
                        DB::table($table)->where('no', $row->no)->update(['slug' => $slug]);
                    } else {
                        DB::table($table)->update(['slug' => $slug]);
                    }
                }
            }

            // --------------------------
            // 3️⃣ DATA INSERTION (Roles, Permissions, Model Roles) inside transaction
            // --------------------------
            DB::beginTransaction();

            // Roles
            $roles = [
                1 => 'Super Admin',
                2 => 'Penyelaras',
                3 => 'Pengguna',
                4 => 'Pengurusan',
            ];
            foreach ($roles as $id => $role) {
                DB::table('roles')->updateOrInsert(
                    ['id' => $id],
                    ['name' => $role, 'guard_name' => 'web', 'created_at' => now(), 'updated_at' => now()]
                );
            }

            // Permissions
            $permissions = [
                'view-any DokumenRujukan', 'view DokumenRujukan', 'create DokumenRujukan', 'update DokumenRujukan', 'delete DokumenRujukan',
                'delete-any DokumenRujukan', 'replicate DokumenRujukan', 'restore DokumenRujukan', 'restore-any DokumenRujukan', 'reorder DokumenRujukan',
            ];

            foreach ($permissions as $permission) {
                DB::table('permissions')->updateOrInsert(
                    ['name' => $permission, 'guard_name' => 'web'],
                    ['created_at' => now(), 'updated_at' => now()]
                );
            }

            // Assign all permissions to Super Admin
            $superAdminId = DB::table('roles')->where('name', 'Super Admin')->value('id');
            $permissionIds = DB::table('permissions')->pluck('id');
            foreach ($permissionIds as $permissionId) {
                DB::table('role_has_permissions')->updateOrInsert([
                    'role_id' => $superAdminId,
                    'permission_id' => $permissionId,
                ]);
            }

            // Assign model_has_roles from users.jenis_pengguna
            $users = DB::table('users')->select('id', 'jenis_pengguna')->get();
            foreach ($users as $user) {
                if (!$user->jenis_pengguna) continue;
                DB::table('model_has_roles')->updateOrInsert([
                    'role_id' => $user->jenis_pengguna,
                    'model_type' => 'App\\Models\\User',
                    'model_id' => $user->id,
                ]);
            }

            DB::commit();

            // Lock installer
            File::put(storage_path('installed.lock'), now());

            return response()->json([
                'status' => 'success',
                'message' => '✅ Installation completed. Installer locked.'
            ]);

        } catch (\Throwable $e) {
            if (DB::transactionLevel() > 0) {
                DB::rollBack();
            }
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ]);
        }
    }
}
