<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PenggunaResource\Pages;
use App\Filament\Resources\PenggunaResource\RelationManagers;
use App\Models\Pengguna;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Columns\TextColumn;
use Illuminate\Support\Facades\Hash;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Tables\Columns\IconColumn;
use Filament\Forms\Components\FileUpload;
use Filament\Tables\Columns\ImageColumn;
use Illuminate\Support\Facades\Storage;
use Filament\Tables\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Auth;
use Filament\Tables\Filters\SelectFilter;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Schema;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

use App\Models\Gred;
use App\Models\JenisPengguna;
use App\Models\InstitusiJabatan;
use App\Models\Jabatan;
use App\Models\Institusi;

class PenggunaResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-s-user-group';

    protected static ?string $navigationLabel = 'PENGGUNA';

    protected static ?string $pluralModelLabel = 'PENGGUNA';

    protected static ?string $navigationGroup = 'UTILITI';

    protected static ?string $slug = 'pengguna';

    public static function form(Form $form): Form
    {
        $jenis_pengguna = auth()->user()->roles()->value('id');
        $email = auth()->user()->email;
        $no_institusi_jabatan = auth()->user()->no_institusi_jabatan;

        return $form->schema([
            Forms\Components\TextInput::make('nama')
                ->label('Nama')
                ->rules(['required', 'max:100']),
            Forms\Components\TextInput::make('email')
                ->email()
                ->label('Emel')
                ->rules(['required', 'max:100']),
            TextInput::make('password')
                ->label('Kata Laluan')
                ->password()
                ->revealable()
                ->autocomplete('new-password')

                // Default password (CREATE sahaja)
                ->default(fn(string $operation) => $operation === 'create' ? 'polimas@2021' : null)

                // Wajib hanya semasa CREATE
                ->required(fn(string $operation) => $operation === 'create')

                // Jangan overwrite password lama jika kosong (EDIT)
                ->dehydrated(fn($state) => filled($state))

                // Hash hanya jika ada input
                ->dehydrateStateUsing(fn($state) => Hash::make($state))

                ->maxLength(100),
            // Forms\Components\TextInput::make('password')->password()->dehydrateStateUsing(fn($state) => Hash::make($state))->label('Kata Laluan')->rules(['required', 'max:100']),
            Select::make('gred')
                ->label('Gred Jawatan')
                ->placeholder('Sila Pilih Maklumat')
                ->options(Gred::all()->pluck('nama', 'no'))
                ->searchable()
                ->rules(['required']),
            Select::make('no_institusi_jabatan')
                ->label('Institusi & Jabatan')
                ->placeholder('Sila Pilih Maklumat')
                ->searchable()
                ->default(function () use ($jenis_pengguna) {
                    if ($jenis_pengguna == 2) {
                        return Auth::user()->no_institusi_jabatan;
                    }
                })
                ->options(function () use ($jenis_pengguna, $no_institusi_jabatan) {
                    // Eager load related models

                    $institusiJabatans = InstitusiJabatan::with('institusi', 'jabatan');

                    if ($jenis_pengguna == 2) {
                        $userId = Auth::id();
                        // Modify the query to include a where clause for jenis_pengguna = 2
                        $institusiJabatans->where('no', $no_institusi_jabatan);
                    }

                    $institusiJabatans = $institusiJabatans->get();

                    // Manipulate data to format as needed for select dropdown
                    return $institusiJabatans->mapWithKeys(function ($institusiJabatan) {
                        return [$institusiJabatan->no => $institusiJabatan->institusi->nama . ' - ' . $institusiJabatan->jabatan->nama];
                    });
                })
                ->rules('required'),
            Select::make('aktif')
                ->placeholder('Sila Pilih Maklumat')
                ->options([
                    '1' => 'AKTIF',
                    '0' => 'TIDAK AKTIF',
                ])
                ->rules(['required']),
            Select::make('roles')
                ->label('Jenis Pengguna')
                ->placeholder('Sila Pilih Maklumat')
                // ->multiple()
                ->relationship('roles', 'name')
                ->options(function () {
                    $user = auth()->user();

                    $query = Role::query()->where('guard_name', 'web');

                    // SUPER ADMIN
                    if ($user->jenis_pengguna == 1) {
                        // boleh lihat semua role
                    }
                    // ADMIN JABATAN
                    elseif ($user->jenis_pengguna == 2) {
                        // contoh: sekat role tertentu
                        $query->whereNotIn('name', ['super-admin']);
                    }

                    return $query->pluck('name', 'id')->toArray();
                })
                ->searchable()
                // ->preload()
                ->required(),
            // Select::make('jenis_pengguna')
            //     ->label('Jenis Pengguna')
            //     ->placeholder('Sila Pilih Maklumat')
            //     ->options(function () use ($jenis_pengguna, $no_institusi_jabatan) {
            //         // Initialize the query for JenisPengguna
            //         $jenisPenggunaQuery = JenisPengguna::query();

            //         // Apply conditions based on $jenis_pengguna
            //         if ($jenis_pengguna == 1) {
            //             // Apply specific conditions for jenis_pengguna = 1
            //             // Example: $jenisPenggunaQuery->where('some_field', 'some_value');
            //         } elseif ($jenis_pengguna == 2) {
            //             // Modify the query to exclude certain values for jenis_pengguna = 2
            //             $jenisPenggunaQuery->whereNotIn('no', [1, 4]);
            //         }

            //         // Fetch the filtered results
            //         $jenisPengguna = $jenisPenggunaQuery->get();

            //         // Map the results to an associative array for the select options
            //         return $jenisPengguna->pluck('nama', 'no')->toArray();
            //     })
            //     ->searchable()
            //     ->rules(['required']),
            // FileUpload::make('gambar')
            //     ->directory('gambar')
            //     ->visibility('private')
        ]);
    }

    public static function table(Table $table): Table
    {
        $jenis_pengguna = Auth::user()->jenis_pengguna;
        $email = Auth::user()->email;
        $no_institusi_jabatan = Auth::user()->no_institusi_jabatan;

        // Adjust the query based on the user's role
        $penggunaQuery = Pengguna::query()->with(['institusiJabatan', 'senaraigred', 'jenisPengguna']);
        if ($jenis_pengguna == 1) {
            // If user is not admin, filter jenis laporan based on user's role
            // You can add your filtering logic here
        } elseif ($jenis_pengguna == 2) {
            // If user is of type 2, filter based on no_institusi_jabatan
            $penggunaQuery->whereHas('institusiJabatan', function ($query) use ($no_institusi_jabatan) {
                $query->where('no_institusi_jabatan', $no_institusi_jabatan);
            });
        } elseif ($jenis_pengguna == 4) {
            // If user is of type 4, exclude specific jenis laporan
            $institusi_jabatan = InstitusiJabatan::where('no', $no_institusi_jabatan)->first();
            $institusi = $institusi_jabatan->no_institusi;

            $penggunaQuery->whereHas('institusiJabatan', function ($query) use ($institusi) {
                $query->where('no_institusi', $institusi);
            });
        }

        return $table
            ->query($penggunaQuery)
            // ✅ TABLE TOPBAR BUTTON
            ->headerActions([
                Action::make('padam_fail_gambar')
                    ->label('Padam Gambar')
                    ->icon('heroicon-o-trash')
                    ->color('danger')
                    ->requiresConfirmation()
                    ->modalHeading('Padam Gambar')
                    ->modalDescription('Fail gambar akan dipadam terus dari sistem.')
                    ->action(function () {
                        $path = 'gambar/01HX6KYTZS1STH9M37NTE4RF0C.jpg';

                        if (Storage::disk('public')->exists($path)) {
                            Storage::disk('public')->delete($path);

                            Notification::make()->title('Gambar berjaya dipadam')->success()->send();
                        } else {
                            Notification::make()->title('Fail tidak dijumpai')->danger()->send();
                        }
                    })

                    // 🔐 ADMIN + FAIL WUJUD BARU PAPAR
                    ->visible(fn() => auth()->user()?->hasRole('Super Admin') && Storage::disk('public')->exists('gambar/01HX6KYTZS1STH9M37NTE4RF0C.jpg')),

                // 🔐 SLUG GENERATOR GLOBAL
                Action::make('slug_generator')
                    ->label('SLUG GENERATOR')
                    ->icon('heroicon-o-key')
                    ->color('warning')
                    ->requiresConfirmation()
                    ->modalHeading('Generate Slug')
                    ->modalDescription('Button ini akan tambah column slug & jana slug sebelum created_at untuk semua rekod.')
                    ->action(function () {
                        $tables = ['dokumen_rujukan', 'gred', 'institusi', 'institusi_jabatan', 'jabatan', 'jawatan', 'jenis_dokumen', 'jenis_dokumen_rujukan', 'jenis_laporan', 'jenis_pengguna', 'menu', 'menu_sub_menu', 'pautan_menu_sub_menu', 'peringkat', 'senarai_jpka', 'sesi', 'status', 'sub_menu', 'tugas', 'tugas_sampingan', 'unit_tugas_sampingan', 'users'];

                        foreach ($tables as $table) {
                            // 1️⃣ Tambah column slug jika tiada
                            if (!Schema::hasColumn($table, 'slug')) {
                                Schema::table($table, function (Blueprint $t) {
                                    $t->string('slug', 150)->nullable();
                                });
                            } else {
                                // 2️⃣ Ubah panjang slug jika perlu
                                Schema::table($table, function (Blueprint $t) {
                                    $t->string('slug', 150)->nullable()->change();
                                });
                            }

                            // 3️⃣ Ambil semua rekod
                            $rows = DB::table($table)->get();

                            foreach ($rows as $row) {
                                // 4️⃣ Generate slug untuk semua rekod tanpa kira primary key
                                $slug = substr(str_replace('-', '', Str::uuid()) . Str::upper(Str::random(68)), 0, 150);

                                // 5️⃣ Update row berdasarkan **primary key dinamika** jika ada, else skip table
                                if (isset($row->id)) {
                                    DB::table($table)
                                        ->where('id', $row->id)
                                        ->update(['slug' => $slug]);
                                } elseif (isset($row->no)) {
                                    DB::table($table)
                                        ->where('no', $row->no)
                                        ->update(['slug' => $slug]);
                                } else {
                                    // Kalau table tiada 'id' atau 'no', kita gunakan **all rows update** (caution!)
                                    DB::table($table)->update(['slug' => $slug]);
                                }
                            }
                        }

                        Notification::make()->title('Slug berjaya dijana')->success()->send();
                    })
                    ->visible(fn() => auth()->user()?->hasRole('Super Admin')),
            ])
            ->columns([
                Tables\Columns\TextColumn::make('index')->label('NO')->rowIndex(),
                Tables\Columns\TextColumn::make('id')->label('ID')->visible(fn() => $jenis_pengguna == 1),
                ImageColumn::make('gambar')
                    ->label('GAMBAR')
                    ->width(150)
                    ->height(150)
                    ->circular()
                    ->getStateUsing(function ($record) {
                        return url('storage/gambar/' . $record->email . '/' . $record->gambar);
                    }),
                Tables\Columns\TextColumn::make('nama')->label('Nama'),
                Tables\Columns\TextColumn::make('email')->label('Emel'),
                Tables\Columns\TextColumn::make('senaraigred.nama')->label('GRED')->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('institusiJabatan.institusi.nama')->label('Institusi'),
                Tables\Columns\TextColumn::make('institusiJabatan.jabatan.nama')->label('Jabatan'),
                Tables\Columns\TextColumn::make('jenisPengguna.nama')->label('Peranan'),
                IconColumn::make('aktif')->label('STATUS')->boolean(),
                TextColumn::make('id')->label('ID')->searchable()->visible(fn() => $jenis_pengguna == 1),
                TextColumn::make('nama')->label('NAMA')->searchable(),
                TextColumn::make('email')->label('EMEL')->searchable(),
                TextColumn::make('institusiJabatan.institusi.nama')->label('INSTITUSI')->searchable()->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('institusiJabatan.jabatan.nama')->label('JABATAN')->searchable()->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('jenisPengguna.nama')->label('JENIS PENGGUNA')->searchable(),
            ])
            ->defaultSort('nama')
            ->deferLoading()
            ->striped()
            ->filters([SelectFilter::make('gred')->relationship('senaraiGred', 'nama')->searchable()->multiple()->preload()])
            ->actions([
                Action::make('muat_turun')
                    ->label('')
                    ->icon('heroicon-o-arrow-up-tray')
                    ->form(function ($record) {
                        $formFields = [];

                        // Assuming $record->no is used to determine the form fields dynamically
                        switch ($record->no) {
                            case 1:
                            //
                            // break;
                            case 2:
                            //
                            // break;
                            case 3:
                            //
                            // break;
                            case 4:
                            //
                            // break;
                            case 5:
                            //
                            // break;
                            // Add more cases as needed

                            default:
                                // Default behavior if $record->no doesn't match any case
                                $formFields[] = FileUpload::make('gambar')
                                    ->label('Muat Naik Gambar')
                                    ->directory(function () use ($record) {
                                        $directory = 'gambar/' . $record->email;
                                        return $directory;
                                    })
                                    ->visibility('private'); // Assuming Sesi is a model

                                break;
                        }

                        return $formFields;
                    })
                    ->action(function (array $data, Pengguna $record) {
                        // Pengguna::where('email', $record->email)->update([
                        //     'gambar' => 1,
                        // ]);

                        // Construct the full path of the file to delete
                        $filePath = storage_path('/app/public/gambar/' . $record->email . '/' . basename($record->gambar));

                        // unlink(storage_path('/app/public/gambar/' . $record->email . '/' . basename($record->gambar)));

                        if (file_exists($filePath)) {
                            // Delete the file
                            unlink(storage_path('/app/public/gambar/' . $record->email . '/' . basename($record->gambar)));
                        }

                        // Find the record using the email
                        $pengguna = Pengguna::where('email', $record->email)->first();

                        // Update the 'gambar' attribute
                        $pengguna->gambar = basename($data['gambar']);

                        // Save the changes
                        $pengguna->save();

                        // Send success notification
                        Notification::make()->title('Muat Naik berjaya.')->success()->send();
                    }),
                Action::make('password')
                    ->label('')
                    ->icon('heroicon-o-cog')
                    ->form(function ($record) {
                        $formFields = [];

                        // Assuming $record->no is used to determine the form fields dynamically
                        switch ($record->no) {
                            case 1:
                            //
                            // break;
                            case 2:
                            //
                            // break;
                            case 3:
                            //
                            // break;
                            case 4:
                            //
                            // break;
                            case 5:
                            //
                            // break;
                            // Add more cases as needed

                            default:
                                // Default behavior if $record->no doesn't match any case
                                $formFields[] = Forms\Components\TextInput::make('password')
                                    ->label('Kata Laluan')
                                    ->rules(['required', 'max:100']); // Assuming Sesi is a model

                                break;
                        }

                        return $formFields;
                    })
                    ->action(function (array $data, Pengguna $record) {
                        // Find the record using the email
                        $pengguna = Pengguna::where('email', $record->email)->first();

                        // Update the 'gambar' attribute
                        $pengguna->password = Hash::make($data['password']);

                        // Save the changes
                        $pengguna->save();

                        // Send success notification
                        Notification::make()->title('Kata Laluan telah dikemaskini.')->success()->send();
                    }),
                Tables\Actions\EditAction::make()->label(''),
                Tables\Actions\DeleteAction::make()->label(''),
            ])
            ->bulkActions([
                // Tables\Actions\BulkActionGroup::make([
                //     Tables\Actions\DeleteBulkAction::make(),
                // ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
                //
            ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPenggunas::route('/'),
            'create' => Pages\CreatePengguna::route('/create'),
            // 'edit' => Pages\EditPengguna::route('/{record}/edit'),
            'edit' => Pages\EditPengguna::route('/{record:slug}/edit'), // use slug here
        ];
    }

    // public static function canViewAny(): bool
    // {
    //     $userId = auth()->user()->jenis_pengguna;
    //     return in_array($userId, [1, 2]);
    // }

    // public static function canViewAny(): bool
    // {
    //     $user = auth()->user();

    //     if (!$user) {
    //         return false;
    //     }

    //     return $user
    //         ->roles()
    //         ->whereIn('id', [1, 2])
    //         ->exists();
    // }

    public static function canViewAny(): bool
    {
        // dd(auth()->user()->roles());
        return auth()
            ->user()
            ->hasAnyRole(['Super Admin', 'Penyelaras']);
    }

    public static function getNavigationBadge(): ?string
    {
        $user = Auth::user();

        if (!$user) {
            return null;
        }

        // Check the user's role and adjust the count query accordingly
        if ($user->jenis_pengguna == 1) {
            // For roles 1 and 2, return the total count
            return static::getModel()::count();
        } elseif ($user->jenis_pengguna == 3) {
            // For role 3, filter by the user's email or institution
            return static::getModel()::where('email', $user->email)->count();
        } elseif (in_array($user->jenis_pengguna, [2, 4])) {
            // For role 4, filter by institution/jabatan
            return static::getModel()::where('no_institusi_jabatan', $user->no_institusi_jabatan)->count();
        }

        // Default to no badge for other roles
        return null;
    }

    public static function canDelete($record): bool
    {
        return !\App\Models\Tugasan::where('email', $record->email)->exists();
    }
}
